<?php
$pageTitle = 'Carte Interactive';
$breadcrumbs = [
    ['label' => 'Dashboard', 'url' => 'index.php'],
    ['label' => 'Carte', 'url' => '#']
];

require_once __DIR__ . '/../config/constants.php';
require_once __DIR__ . '/../config/functions.php';

// Charger les données mock
$interventionsData = json_decode(file_get_contents(__DIR__ . '/../mockdata/interventions.json'), true);
$mechanicsData = json_decode(file_get_contents(__DIR__ . '/../mockdata/mechanics.json'), true);

$interventions = $interventionsData['interventions'];
$mechanics = $mechanicsData['mechanics'];

// Statistiques pour la carte
$activeInterventions = array_filter($interventions, function($int) {
    return in_array($int['status'], ['pending', 'accepted', 'in_progress']);
});
$onlineMechanics = array_filter($mechanics, function($mech) {
    return $mech['status'] === 'online';
});
?>

<!-- Page Header -->
<div class="page-header">
    <div class="page-header-content">
        <h1 class="page-title">
            <i class="bi bi-geo-alt-fill"></i>
            Carte Interactive
        </h1>
        <p class="page-subtitle">Suivi en temps réel des interventions et mécaniciens</p>
    </div>
    <div class="page-header-actions">
        <div class="map-stats">
            <div class="stat-item">
                <span class="stat-icon bg-success">
                    <i class="bi bi-tools"></i>
                </span>
                <div class="stat-info">
                    <span class="stat-value"><?= count($activeInterventions) ?></span>
                    <span class="stat-label">Interventions actives</span>
                </div>
            </div>
            <div class="stat-item">
                <span class="stat-icon bg-primary">
                    <i class="bi bi-person-check"></i>
                </span>
                <div class="stat-info">
                    <span class="stat-value"><?= count($onlineMechanics) ?></span>
                    <span class="stat-label">Mécaniciens en ligne</span>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Map Container -->
<div class="row">
    <div class="col-12">
        <div class="dashboard-card">
            <div class="card-header">
                <div class="card-header-content">
                    <h5 class="card-title">
                        <i class="bi bi-map"></i>
                        Carte des Interventions
                    </h5>
                    <p class="card-subtitle">Cameroun - Zone de couverture MECASOS</p>
                </div>
                <div class="card-header-actions">
                    <!-- Layer Controls -->
                    <div class="btn-group" role="group">
                        <button type="button" class="btn btn-sm btn-outline-secondary layer-toggle active" data-layer="mechanics">
                            <i class="bi bi-person"></i>
                            Mécaniciens
                        </button>
                        <button type="button" class="btn btn-sm btn-outline-secondary layer-toggle active" data-layer="interventions">
                            <i class="bi bi-tools"></i>
                            Interventions
                        </button>
                        <button type="button" class="btn btn-sm btn-outline-secondary layer-toggle" data-layer="heatmap">
                            <i class="bi bi-fire"></i>
                            Zones chaudes
                        </button>
                    </div>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="map-container" id="mainMap" style="height: 600px;">
                    <!-- Google Maps sera intégré ici -->
                    <div class="map-placeholder">
                        <div class="map-placeholder-content">
                            <i class="bi bi-map" style="font-size: 4rem; color: var(--text-secondary);"></i>
                            <h4>Carte Google Maps</h4>
                            <p class="text-muted">Configuration de l'API Google Maps requise</p>
                            <div class="map-placeholder-actions">
                                <button class="btn btn-primary" onclick="initializeMap()">
                                    <i class="bi bi-play-circle"></i>
                                    Initialiser la carte
                                </button>
                                <small class="text-muted d-block mt-2">
                                    Nécessite une clé API Google Maps valide
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="card-footer">
                <!-- Map Legend -->
                <div class="map-legend">
                    <div class="legend-item">
                        <span class="legend-marker mechanic-online"></span>
                        <span class="legend-label">Mécanicien en ligne</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-marker mechanic-busy"></span>
                        <span class="legend-label">Mécanicien occupé</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-marker mechanic-offline"></span>
                        <span class="legend-label">Mécanicien hors ligne</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-marker intervention-pending"></span>
                        <span class="legend-label">Intervention en attente</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-marker intervention-progress"></span>
                        <span class="legend-label">Intervention en cours</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-marker intervention-completed"></span>
                        <span class="legend-label">Intervention terminée</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Map Controls Sidebar -->
<div class="map-sidebar" id="mapSidebar">
    <div class="map-sidebar-header">
        <h6>Détails</h6>
        <button class="btn-close" onclick="closeMapSidebar()"></button>
    </div>
    <div class="map-sidebar-content" id="sidebarContent">
        <div class="sidebar-placeholder">
            <i class="bi bi-info-circle"></i>
            <p>Cliquez sur un marqueur pour voir les détails</p>
        </div>
    </div>
</div>

<!-- Search & Filters -->
<div class="row mt-4">
    <div class="col-xl-8">
        <div class="dashboard-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-search"></i>
                    Recherche & Filtres
                </h5>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label class="form-label">Rechercher une localisation</label>
                            <div class="input-group">
                                <input type="text" class="form-control" id="locationSearch" placeholder="Douala, Yaoundé...">
                                <button class="btn btn-outline-secondary" type="button">
                                    <i class="bi bi-search"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label class="form-label">Statut interventions</label>
                            <select class="form-select" id="interventionFilter">
                                <option value="all">Toutes</option>
                                <option value="pending">En attente</option>
                                <option value="in_progress">En cours</option>
                                <option value="completed">Terminées</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label class="form-label">Statut mécaniciens</label>
                            <select class="form-select" id="mechanicFilter">
                                <option value="all">Tous</option>
                                <option value="online">En ligne</option>
                                <option value="busy">Occupés</option>
                                <option value="offline">Hors ligne</option>
                            </select>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-4">
        <div class="dashboard-card">
            <div class="card-header">
                <h5 class="card-title">
                    <i class="bi bi-clock-history"></i>
                    Activité récente
                </h5>
            </div>
            <div class="card-body">
                <div class="activity-list">
                    <?php foreach(array_slice($interventions, 0, 5) as $intervention): ?>
                    <div class="activity-item">
                        <div class="activity-icon">
                            <i class="bi bi-tools"></i>
                        </div>
                        <div class="activity-content">
                            <div class="activity-title">
                                Intervention <?= $intervention['id'] ?>
                            </div>
                            <div class="activity-meta">
                                <?= e($intervention['client']) ?> • <?= e($intervention['location']) ?>
                            </div>
                            <div class="activity-time">
                                <?= timeAgo($intervention['created_at']) ?>
                            </div>
                        </div>
                        <div class="activity-status">
                            <?= getStatusBadge($intervention['status']) ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// ========================================
// GOOGLE MAPS INTEGRATION
// ========================================
let map;
let markers = [];
let heatmap = null;

// Configuration des icônes
const markerIcons = {
    mechanic_online: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="40" height="40" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg">
                <circle cx="20" cy="20" r="18" fill="#10B981" stroke="#1D1D1B" stroke-width="3"/>
                <circle cx="20" cy="15" r="6" fill="#1D1D1B"/>
                <path d="M10 35 Q20 25 30 35" stroke="#1D1D1B" stroke-width="3" fill="none"/>
            </svg>
        `),
        scaledSize: new google.maps.Size(40, 40)
    },
    mechanic_busy: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="40" height="40" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg">
                <circle cx="20" cy="20" r="18" fill="#F59E0B" stroke="#1D1D1B" stroke-width="3"/>
                <circle cx="20" cy="15" r="6" fill="#1D1D1B"/>
                <path d="M10 35 Q20 25 30 35" stroke="#1D1D1B" stroke-width="3" fill="none"/>
            </svg>
        `),
        scaledSize: new google.maps.Size(40, 40)
    },
    mechanic_offline: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="40" height="40" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg">
                <circle cx="20" cy="20" r="18" fill="#6B7280" stroke="#1D1D1B" stroke-width="3"/>
                <circle cx="20" cy="15" r="6" fill="#1D1D1B"/>
                <path d="M10 35 Q20 25 30 35" stroke="#1D1D1B" stroke-width="3" fill="none"/>
            </svg>
        `),
        scaledSize: new google.maps.Size(40, 40)
    },
    intervention_pending: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="30" height="30" viewBox="0 0 30 30" xmlns="http://www.w3.org/2000/svg">
                <circle cx="15" cy="15" r="13" fill="#F59E0B" stroke="#1D1D1B" stroke-width="2"/>
                <text x="15" y="20" text-anchor="middle" fill="#1D1D1B" font-size="16" font-weight="bold">!</text>
            </svg>
        `),
        scaledSize: new google.maps.Size(30, 30)
    },
    intervention_progress: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="30" height="30" viewBox="0 0 30 30" xmlns="http://www.w3.org/2000/svg">
                <circle cx="15" cy="15" r="13" fill="#3B82F6" stroke="#1D1D1B" stroke-width="2"/>
                <text x="15" y="20" text-anchor="middle" fill="#FFFFFF" font-size="12" font-weight="bold">●</text>
            </svg>
        `),
        scaledSize: new google.maps.Size(30, 30)
    },
    intervention_completed: {
        url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="30" height="30" viewBox="0 0 30 30" xmlns="http://www.w3.org/2000/svg">
                <circle cx="15" cy="15" r="13" fill="#10B981" stroke="#1D1D1B" stroke-width="2"/>
                <text x="15" y="19" text-anchor="middle" fill="#FFFFFF" font-size="14" font-weight="bold">✓</text>
            </svg>
        `),
        scaledSize: new google.maps.Size(30, 30)
    }
};

// Données PHP vers JavaScript
const interventions = <?= json_encode($interventions) ?>;
const mechanics = <?= json_encode($mechanics) ?>;

function initializeMap() {
    // Vérifier si l'API Google Maps est chargée
    if (typeof google === 'undefined') {
        // Charger l'API Google Maps (remplacer YOUR_API_KEY par votre clé)
        const script = document.createElement('script');
        script.src = 'https://maps.googleapis.com/maps/api/js?key=YOUR_GOOGLE_MAPS_API_KEY&callback=initMap';
        script.async = true;
        script.defer = true;
        document.head.appendChild(script);

        // Fonction de callback
        window.initMap = function() {
            createMap();
        };
    } else {
        createMap();
    }
}

function createMap() {
    // Configuration de la carte centrée sur le Cameroun
    const mapOptions = {
        center: { lat: 3.8667, lng: 11.5167 }, // Yaoundé, Cameroun
        zoom: 7,
        styles: [
            {
                "featureType": "all",
                "elementType": "geometry",
                "stylers": [{ "color": "#1d1d1d" }]
            },
            {
                "featureType": "all",
                "elementType": "labels.text.stroke",
                "stylers": [{ "color": "#1d1d1d" }]
            },
            {
                "featureType": "all",
                "elementType": "labels.text.fill",
                "stylers": [{ "color": "#e1a004" }]
            }
        ],
        disableDefaultUI: false,
        zoomControl: true,
        mapTypeControl: false,
        scaleControl: true,
        streetViewControl: false,
        rotateControl: false,
        fullscreenControl: true
    };

    // Créer la carte
    map = new google.maps.Map(document.getElementById('mainMap'), mapOptions);

    // Ajouter les marqueurs
    addMarkers();

    // Masquer le placeholder
    document.querySelector('.map-placeholder').style.display = 'none';
}

function addMarkers() {
    // Vider les marqueurs existants
    markers.forEach(marker => marker.setMap(null));
    markers = [];

    // Filtrer selon les contrôles de couche
    const showMechanics = document.querySelector('[data-layer="mechanics"]').classList.contains('active');
    const showInterventions = document.querySelector('[data-layer="interventions"]').classList.contains('active');

    // Ajouter les mécaniciens
    if (showMechanics) {
        mechanics.forEach(mechanic => {
            if (mechanic.coordinates) {
                const marker = new google.maps.Marker({
                    position: mechanic.coordinates,
                    map: map,
                    icon: getMechanicIcon(mechanic.status),
                    title: mechanic.name,
                    mechanic: mechanic
                });

                marker.addListener('click', () => showMechanicDetails(mechanic));
                markers.push(marker);
            }
        });
    }

    // Ajouter les interventions
    if (showInterventions) {
        interventions.forEach(intervention => {
            if (intervention.coordinates) {
                const marker = new google.maps.Marker({
                    position: intervention.coordinates,
                    map: map,
                    icon: getInterventionIcon(intervention.status),
                    title: `Intervention ${intervention.id}`,
                    intervention: intervention
                });

                marker.addListener('click', () => showInterventionDetails(intervention));
                markers.push(marker);
            }
        });
    }
}

function getMechanicIcon(status) {
    switch(status) {
        case 'online': return markerIcons.mechanic_online;
        case 'busy': return markerIcons.mechanic_busy;
        default: return markerIcons.mechanic_offline;
    }
}

function getInterventionIcon(status) {
    switch(status) {
        case 'pending': return markerIcons.intervention_pending;
        case 'accepted':
        case 'in_progress': return markerIcons.intervention_progress;
        case 'completed': return markerIcons.intervention_completed;
        default: return markerIcons.intervention_pending;
    }
}

function showMechanicDetails(mechanic) {
    const sidebar = document.getElementById('mapSidebar');
    const content = document.getElementById('sidebarContent');

    content.innerHTML = `
        <div class="mechanic-details">
            <div class="mechanic-header">
                <img src="${mechanic.avatar || '<?= ASSETS_URL ?>img/avatars/default.png'}" alt="" class="mechanic-avatar">
                <div class="mechanic-info">
                    <h6>${mechanic.name}</h6>
                    <span class="status-badge status-${mechanic.status}">${mechanic.status}</span>
                </div>
            </div>
            <div class="mechanic-stats">
                <div class="stat-row">
                    <span class="stat-label">Téléphone:</span>
                    <span class="stat-value">${mechanic.phone}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Localisation:</span>
                    <span class="stat-value">${mechanic.location}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Note:</span>
                    <span class="stat-value">⭐ ${mechanic.rating}</span>
                </div>
                <div class="stat-row">
                    <span class="stat-label">Interventions:</span>
                    <span class="stat-value">${mechanic.total_interventions}</span>
                </div>
            </div>
            <div class="mechanic-actions">
                <button class="btn btn-primary btn-sm w-100" onclick="contactMechanic('${mechanic.id}')">
                    <i class="bi bi-chat"></i>
                    Contacter
                </button>
            </div>
        </div>
    `;

    sidebar.classList.add('active');
}

function showInterventionDetails(intervention) {
    const sidebar = document.getElementById('mapSidebar');
    const content = document.getElementById('sidebarContent');

    content.innerHTML = `
        <div class="intervention-details">
            <div class="intervention-header">
                <div class="intervention-id">ID: ${intervention.id}</div>
                <span class="status-badge status-${intervention.status}">${intervention.status.replace('_', ' ')}</span>
            </div>
            <div class="intervention-info">
                <div class="info-row">
                    <span class="info-label">Client:</span>
                    <span class="info-value">${intervention.client}</span>
                </div>
                <div class="info-row">
                    <span class="info-label">Mécanicien:</span>
                    <span class="info-value">${intervention.mechanic}</span>
                </div>
                <div class="info-row">
                    <span class="info-label">Type:</span>
                    <span class="info-value">${intervention.type}</span>
                </div>
                <div class="info-row">
                    <span class="info-label">Montant:</span>
                    <span class="info-value">${intervention.amount.toLocaleString('fr-FR')} FCFA</span>
                </div>
                <div class="info-row">
                    <span class="info-label">Localisation:</span>
                    <span class="info-value">${intervention.location}</span>
                </div>
            </div>
            <div class="intervention-actions">
                <button class="btn btn-primary btn-sm w-100" onclick="viewInterventionDetails('${intervention.id}')">
                    <i class="bi bi-eye"></i>
                    Voir détails
                </button>
            </div>
        </div>
    `;

    sidebar.classList.add('active');
}

function closeMapSidebar() {
    document.getElementById('mapSidebar').classList.remove('active');
}

function contactMechanic(mechanicId) {
    // Implémentation du contact mécanicien
    console.log('Contacter mécanicien:', mechanicId);
}

function viewInterventionDetails(interventionId) {
    // Redirection vers la page de détails
    window.location.href = `interventions/details.php?id=${interventionId}`;
}

// Contrôles des couches
document.addEventListener('DOMContentLoaded', function() {
    // Gestion des boutons de couche
    document.querySelectorAll('.layer-toggle').forEach(button => {
        button.addEventListener('click', function() {
            this.classList.toggle('active');
            addMarkers(); // Recharger les marqueurs
        });
    });

    // Filtres
    document.getElementById('interventionFilter').addEventListener('change', addMarkers);
    document.getElementById('mechanicFilter').addEventListener('change', addMarkers);

    // Recherche
    document.getElementById('locationSearch').addEventListener('input', function() {
        // Implémentation de la recherche à ajouter
        console.log('Recherche:', this.value);
    });
});
</script>

<style>
/* Map Styles */
.map-container {
    position: relative;
    border-radius: 8px;
    overflow: hidden;
}

.map-placeholder {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, #1d1d1d 0%, #2d2d2d 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10;
}

.map-placeholder-content {
    text-align: center;
    color: var(--text-secondary);
}

.map-placeholder-actions {
    margin-top: 20px;
}

/* Map Sidebar */
.map-sidebar {
    position: fixed;
    top: 0;
    right: -400px;
    width: 400px;
    height: 100vh;
    background: var(--card-background);
    border-left: 1px solid var(--border-color);
    transition: right 0.3s ease;
    z-index: 1000;
    overflow-y: auto;
}

.map-sidebar.active {
    right: 0;
}

.map-sidebar-header {
    padding: 20px;
    border-bottom: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.map-sidebar-content {
    padding: 20px;
}

/* Legend */
.map-legend {
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 8px;
}

.legend-marker {
    width: 16px;
    height: 16px;
    border-radius: 50%;
}

.legend-marker.mechanic-online { background: #10B981; }
.legend-marker.mechanic-busy { background: #F59E0B; }
.legend-marker.mechanic-offline { background: #6B7280; }
.legend-marker.intervention-pending { background: #F59E0B; }
.legend-marker.intervention-progress { background: #3B82F6; }
.legend-marker.intervention-completed { background: #10B981; }

/* Stats */
.map-stats {
    display: flex;
    gap: 20px;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 10px;
}

.stat-icon {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
}

.stat-info {
    display: flex;
    flex-direction: column;
}

.stat-value {
    font-size: 18px;
    font-weight: bold;
    color: var(--text-primary);
}

.stat-label {
    font-size: 12px;
    color: var(--text-secondary);
}

/* Activity List */
.activity-list {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.activity-item {
    display: flex;
    align-items: flex-start;
    gap: 12px;
    padding: 12px;
    border-radius: 8px;
    background: rgba(255, 255, 255, 0.05);
}

.activity-icon {
    width: 32px;
    height: 32px;
    border-radius: 6px;
    background: var(--primary-color);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    flex-shrink: 0;
}

.activity-content {
    flex: 1;
}

.activity-title {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 4px;
}

.activity-meta {
    font-size: 12px;
    color: var(--text-secondary);
    margin-bottom: 2px;
}

.activity-time {
    font-size: 11px;
    color: var(--text-muted);
}

.activity-status {
    flex-shrink: 0;
}

/* Responsive */
@media (max-width: 768px) {
    .map-sidebar {
        width: 100vw;
        right: -100vw;
    }

    .map-stats {
        flex-direction: column;
        gap: 10px;
    }

    .map-legend {
        justify-content: center;
    }
}
</style>
