/**
 * MECASOS Admin Dashboard - Users Management
 * Gestion avancée des utilisateurs clients
 */

// Données des clients (simulées - à remplacer par API)
let clientsData = [];
let selectedClients = [];
let dataTable;

document.addEventListener('DOMContentLoaded', function() {
    
    // ========================================
    // INITIALIZATION
    // ========================================
    
    initDataTable();
    initFilters();
    initCheckboxes();
    initFormValidation();
    
    // ========================================
    // DATATABLES INITIALIZATION
    // ========================================
    
    function initDataTable() {
        dataTable = $('#clientsTable').DataTable({
            responsive: true,
            processing: true,
            language: {
                url: '//cdn.datatables.net/plug-ins/1.13.7/i18n/fr-FR.json',
                processing: '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Chargement...</span></div>'
            },
            pageLength: 15,
            lengthMenu: [[10, 15, 25, 50, -1], [10, 15, 25, 50, "Tous"]],
            order: [[11, 'desc']], // Trier par dernière activité
            columnDefs: [
                { 
                    orderable: false, 
                    targets: [0, 12] // Checkbox et Actions
                },
                {
                    targets: [7, 8], // Dépenses et Wallet
                    render: function(data, type, row) {
                        if (type === 'sort' || type === 'type') {
                            return parseFloat(data.replace(/[^\d]/g, ''));
                        }
                        return data;
                    }
                }
            ],
            dom: '<"row"<"col-sm-12 col-md-6"l><"col-sm-12 col-md-6"f>>rt<"row"<"col-sm-12 col-md-5"i><"col-sm-12 col-md-7"p>>',
            buttons: [
                {
                    extend: 'excelHtml5',
                    text: '<i class="bi bi-file-excel"></i> Excel',
                    className: 'btn btn-sm btn-success',
                    title: 'Clients MECASOS - ' + new Date().toLocaleDateString('fr-FR'),
                    exportOptions: {
                        columns: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]
                    }
                },
                {
                    extend: 'pdfHtml5',
                    text: '<i class="bi bi-file-pdf"></i> PDF',
                    className: 'btn btn-sm btn-danger',
                    title: 'Clients MECASOS',
                    orientation: 'landscape',
                    exportOptions: {
                        columns: [1, 2, 3, 4, 5, 6]
                    }
                },
                {
                    extend: 'csvHtml5',
                    text: '<i class="bi bi-file-csv"></i> CSV',
                    className: 'btn btn-sm btn-info',
                    exportOptions: {
                        columns: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]
                    }
                },
                {
                    extend: 'print',
                    text: '<i class="bi bi-printer"></i> Imprimer',
                    className: 'btn btn-sm btn-secondary',
                    exportOptions: {
                        columns: [1, 2, 3, 4, 5, 6]
                    }
                }
            ],
            drawCallback: function() {
                // Réinitialiser les checkboxes après le redraw
                updateCheckboxStates();
            },
            initComplete: function() {
                console.log('DataTable initialisée avec succès');
                
                // Animation d'entrée des lignes
                animateTableRows();
            }
        });
        
        // Cacher les boutons par défaut (on les affichera dans la fonction exportData)
        $('.dt-buttons').hide();
    }
    
    // ========================================
    // FILTERS MANAGEMENT
    // ========================================
    
    function initFilters() {
        // Recherche globale
        $('#globalSearch').on('keyup', debounce(function() {
            dataTable.search(this.value).draw();
        }, 300));
        
        // Filtre par statut
        $('#statusFilter').on('change', function() {
            const status = this.value;
            dataTable.column(4).search(status).draw();
        });
        
        // Filtre par vérification
        $('#verificationFilter').on('change', function() {
            const verification = this.value;
            dataTable.column(5).search(verification).draw();
        });
        
        // Réinitialiser les filtres
        $('#resetFilters').on('click', function() {
            $('#globalSearch').val('');
            $('#statusFilter').val('');
            $('#verificationFilter').val('');
            $('#dateRangePicker').val('');
            dataTable.search('').columns().search('').draw();
            showToast('Filtres réinitialisés', 'info');
        });
        
        // Actualiser le tableau
        $('#refreshTable').on('click', function() {
            const btn = $(this);
            const icon = btn.find('i');
            
            icon.addClass('rotate-animation');
            dataTable.ajax.reload(function() {
                icon.removeClass('rotate-animation');
                showToast('Tableau actualisé', 'success');
            }, false);
        });
    }
    
    // ========================================
    // CHECKBOXES MANAGEMENT
    // ========================================
    
    function initCheckboxes() {
        // Select All
        $('#selectAll').on('change', function() {
            const isChecked = this.checked;
            $('.row-checkbox:visible').prop('checked', isChecked);
            updateSelectedClients();
        });
        
        // Individual checkboxes
        $(document).on('change', '.row-checkbox', function() {
            updateSelectedClients();
            
            // Update "Select All" state
            const totalVisible = $('.row-checkbox:visible').length;
            const totalChecked = $('.row-checkbox:checked').length;
            $('#selectAll').prop('checked', totalVisible === totalChecked && totalVisible > 0);
        });
    }
    
    function updateSelectedClients() {
        selectedClients = [];
        $('.row-checkbox:checked').each(function() {
            selectedClients.push(parseInt($(this).val()));
        });
        
        // Afficher/masquer la barre d'actions en masse
        if (selectedClients.length > 0) {
            $('#bulkActionsBar').slideDown(300);
            $('#selectedCount').text(selectedClients.length);
        } else {
            $('#bulkActionsBar').slideUp(300);
        }
    }
    
    function updateCheckboxStates() {
        $('.row-checkbox').each(function() {
            const clientId = parseInt($(this).val());
            $(this).prop('checked', selectedClients.includes(clientId));
        });
    }
    
    // ========================================
    // FORM VALIDATION
    // ========================================
    
    function initFormValidation() {
        const form = document.getElementById('addClientForm');
        
        if (form) {
            form.addEventListener('submit', function(e) {
                e.preventDefault();
                
                // Validation personnalisée
                const password = form.querySelector('[name="password"]').value;
                const passwordConfirm = form.querySelector('[name="password_confirm"]').value;
                
                if (password !== passwordConfirm) {
                    showToast('Les mots de passe ne correspondent pas', 'error');
                    return false;
                }
                
                // Validation email
                const email = form.querySelector('[name="email"]').value;
                if (!isValidEmail(email)) {
                    showToast('Email invalide', 'error');
                    return false;
                }
                
                // Validation téléphone
                const phone = form.querySelector('[name="phone"]').value;
                if (!isValidPhone(phone)) {
                    showToast('Numéro de téléphone invalide', 'error');
                    return false;
                }
                
                // Si tout est OK, soumettre
                submitAddClientForm(new FormData(form));
            });
        }
    }
    
    function submitAddClientForm(formData) {
        toggleLoading(true);
        
        // Simulation d'appel API
        setTimeout(() => {
            toggleLoading(false);
            
            // Fermer la modale
            const modal = bootstrap.Modal.getInstance(document.getElementById('addClientModal'));
            modal.hide();
            
            // Reset form
            document.getElementById('addClientForm').reset();
            
            // Recharger le tableau
            dataTable.ajax.reload(null, false);
            
            showToast('Client créé avec succès !', 'success');
        }, 1500);
    }
    
    // ========================================
    // ANIMATION UTILITIES
    // ========================================
    
    function animateTableRows() {
        $('.users-table tbody tr').each(function(index) {
            $(this).css({
                opacity: 0,
                transform: 'translateY(20px)'
            }).delay(index * 50).animate({
                opacity: 1
            }, 400, function() {
                $(this).css('transform', 'translateY(0)');
            });
        });
    }
    
    // Debounce utility
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    // ========================================
    // VALIDATION UTILITIES
    // ========================================
    
    function isValidEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
    
    function isValidPhone(phone) {
        const re = /^\+?237\s?[67]\d{8}$/;
        return re.test(phone.replace(/\s/g, ''));
    }
    
});

// ============================================
// CLIENT ACTIONS (Global Functions)
// ============================================

/**
 * Voir les détails d'un client
 */
function viewClient(clientId) {
    toggleLoading(true);
    
    // Simulation de chargement de données
    setTimeout(() => {
        const modalContent = generateClientDetailsHTML(clientId);
        document.getElementById('clientDetailsContent').innerHTML = modalContent;
        
        const modal = new bootstrap.Modal(document.getElementById('viewClientModal'));
        modal.show();
        
        toggleLoading(false);
    }, 800);
}

function generateClientDetailsHTML(clientId) {
    // Mock data - à remplacer par un vrai appel API
    const client = {
        id: clientId,
        name: "Jean Kouam",
        email: "jean.kouam@example.com",
        phone: "+237 690 123 456",
        status: "active",
        verification_status: "verified",
        location: "Douala, Akwa",
        registration_date: "2024-05-15 10:30:00",
        last_login: "2025-11-21 07:15:00",
        total_interventions: 47,
        total_spent: 890000,
        wallet_balance: 25000,
        rating: 4.8,
        loyalty_points: 450,
        referrals: 3,
        vehicles: [
            { brand: "Toyota", model: "Corolla", year: 2019, plate: "CE-2345-BA" },
            { brand: "Honda", model: "Civic", year: 2021, plate: "CE-7890-XY" }
        ]
    };
    
    return `
        <div class="client-details-wrapper">
            <!-- Header Section -->
            <div class="client-details-header">
                <div class="client-avatar-large">
                    <img src="https://ui-avatars.com/api/?name=${encodeURIComponent(client.name)}&background=E1A004&color=1D1D1B&bold=true&size=128" alt="${client.name}">
                    <div class="status-badge status-${client.status}">
                        <i class="bi bi-circle-fill"></i>
                        ${client.status === 'active' ? 'Actif' : client.status === 'suspended' ? 'Suspendu' : 'Inactif'}
                    </div>
                </div>
                <div class="client-header-info">
                    <h3>${client.name}</h3>
                    <div class="client-meta">
                        <span class="meta-item">
                            <i class="bi bi-shield-check text-success"></i>
                            Compte vérifié
                        </span>
                        <span class="meta-item">
                            <i class="bi bi-calendar3"></i>
                            Membre depuis ${new Date(client.registration_date).toLocaleDateString('fr-FR')}
                        </span>
                        <span class="meta-item">
                            <i class="bi bi-star-fill text-warning"></i>
                            ${client.rating}/5
                        </span>
                    </div>
                </div>
            </div>
            
            <!-- Stats Row -->
            <div class="client-stats-grid">
                <div class="stat-box">
                    <div class="stat-icon bg-primary">
                        <i class="bi bi-tools"></i>
                    </div>
                    <div class="stat-data">
                        <span class="stat-value">${client.total_interventions}</span>
                        <span class="stat-label">Interventions</span>
                    </div>
                </div>
                <div class="stat-box">
                    <div class="stat-icon bg-success">
                        <i class="bi bi-currency-dollar"></i>
                    </div>
                    <div class="stat-data">
                        <span class="stat-value">${formatCurrency(client.total_spent)}</span>
                        <span class="stat-label">Dépenses totales</span>
                    </div>
                </div>
                <div class="stat-box">
                    <div class="stat-icon bg-warning">
                        <i class="bi bi-wallet2"></i>
                    </div>
                    <div class="stat-data">
                        <span class="stat-value">${formatCurrency(client.wallet_balance)}</span>
                        <span class="stat-label">Solde wallet</span>
                    </div>
                </div>
                <div class="stat-box">
                    <div class="stat-icon bg-info">
                        <i class="bi bi-trophy"></i>
                    </div>
                    <div class="stat-data">
                        <span class="stat-value">${client.loyalty_points}</span>
                        <span class="stat-label">Points fidélité</span>
                    </div>
                </div>
            </div>
            
            <!-- Tabs Section -->
            <ul class="nav nav-tabs client-tabs" id="clientTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#info-tab">
                        <i class="bi bi-info-circle"></i> Informations
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" data-bs-toggle="tab" data-bs-target="#vehicles-tab">
                        <i class="bi bi-car-front"></i> Véhicules (${client.vehicles.length})
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" data-bs-toggle="tab" data-bs-target="#history-tab">
                        <i class="bi bi-clock-history"></i> Historique
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" data-bs-toggle="tab" data-bs-target="#activity-tab">
                        <i class="bi bi-activity"></i> Activité
                    </button>
                </li>
            </ul>
            
            <div class="tab-content client-tab-content">
                <!-- Informations Tab -->
                <div class="tab-pane fade show active" id="info-tab">
                    <div class="info-grid">
                        <div class="info-item">
                            <i class="bi bi-envelope"></i>
                            <div>
                                <span class="info-label">Email</span>
                                <span class="info-value">${client.email}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-phone"></i>
                            <div>
                                <span class="info-label">Téléphone</span>
                                <span class="info-value">${client.phone}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-geo-alt"></i>
                            <div>
                                <span class="info-label">Localisation</span>
                                <span class="info-value">${client.location}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-calendar-check"></i>
                            <div>
                                <span class="info-label">Dernière connexion</span>
                                <span class="info-value">${timeAgo(client.last_login)}</span>
                            </div>
                        </div>
                        <div class="info-item">
                            <i class="bi bi-people"></i>
                            <div>
                                <span class="info-label">Parrainages</span>
                                <span class="info-value">${client.referrals} client(s) parrainé(s)</span>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Vehicles Tab -->
                <div class="tab-pane fade" id="vehicles-tab">
                    <div class="vehicles-list">
                        ${client.vehicles.map((vehicle, index) => `
                            <div class="vehicle-card">
                                <div class="vehicle-icon">
                                    <i class="bi bi-car-front-fill"></i>
                                </div>
                                <div class="vehicle-info">
                                    <h6 class="vehicle-title">${vehicle.brand} ${vehicle.model}</h6>
                                    <div class="vehicle-details">
                                        <span><i class="bi bi-calendar3"></i> ${vehicle.year}</span>
                                        <span><i class="bi bi-card-text"></i> ${vehicle.plate}</span>
                                    </div>
                                </div>
                                <button class="btn btn-sm btn-outline-primary" onclick="editVehicle(${clientId}, ${index})">
                                    <i class="bi bi-pencil"></i>
                                </button>
                            </div>
                        `).join('')}
                        <button class="btn btn-outline-primary w-100 mt-3" onclick="addVehicle(${clientId})">
                            <i class="bi bi-plus-lg"></i>
                            Ajouter un véhicule
                        </button>
                    </div>
                </div>
                
                <!-- History Tab -->
                <div class="tab-pane fade" id="history-tab">
                    <div class="timeline">
                        <div class="timeline-item">
                            <div class="timeline-marker bg-success"></div>
                            <div class="timeline-content">
                                <h6>Intervention terminée</h6>
                                <p>Changement de batterie - 15,000 FCFA</p>
                                <small class="text-muted">Il y a 2 heures</small>
                            </div>
                        </div>
                        <div class="timeline-item">
                            <div class="timeline-marker bg-primary"></div>
                            <div class="timeline-content">
                                <h6>Recharge wallet</h6>
                                <p>+25,000 FCFA</p>
                                <small class="text-muted">Il y a 1 jour</small>
                            </div>
                        </div>
                        <div class="timeline-item">
                            <div class="timeline-marker bg-info"></div>
                            <div class="timeline-content">
                                <h6>Compte vérifié</h6>
                                <p>Documents validés avec succès</p>
                                <small class="text-muted">Il y a 3 jours</small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Activity Tab -->
                <div class="tab-pane fade" id="activity-tab">
                    <div class="activity-list">
                        <div class="activity-item">
                            <i class="bi bi-box-arrow-in-right text-success"></i>
                            <div>
                                <p>Connexion depuis <strong>Mobile App</strong></p>
                                <small class="text-muted">Aujourd'hui à 07:15</small>
                            </div>
                        </div>
                        <div class="activity-item">
                            <i class="bi bi-star-fill text-warning"></i>
                            <div>
                                <p>A noté un mécanicien <strong>5/5</strong></p>
                                <small class="text-muted">Hier à 18:30</small>
                            </div>
                        </div>
                        <div class="activity-item">
                            <i class="bi bi-wallet2 text-primary"></i>
                            <div>
                                <p>Recharge wallet de <strong>25,000 FCFA</strong></p>
                                <small class="text-muted">Il y a 2 jours</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Action Buttons -->
            <div class="client-actions-footer">
                <button class="btn btn-outline-secondary" onclick="sendNotification(${clientId})">
                    <i class="bi bi-send"></i>
                    Envoyer notification
                </button>
                <button class="btn btn-outline-primary" onclick="editClient(${clientId})">
                    <i class="bi bi-pencil"></i>
                    Modifier
                </button>
                <button class="btn btn-outline-warning" onclick="suspendClient(${clientId})">
                    <i class="bi bi-ban"></i>
                    Suspendre
                </button>
                <button class="btn btn-outline-danger" onclick="deleteClient(${clientId})">
                    <i class="bi bi-trash"></i>
                    Supprimer
                </button>
            </div>
        </div>
    `;
}

/**
 * Éditer un client
 */
function editClient(clientId) {
    console.log('Éditer client:', clientId);
    showToast('Fonctionnalité d\'édition en cours de développement', 'info');
}

/**
 * Supprimer un client
 */
function deleteClient(clientId) {
    if (confirm('Êtes-vous sûr de vouloir supprimer ce client ? Cette action est irréversible.')) {
        toggleLoading(true);
        
        // Simulation suppression
        setTimeout(() => {
            toggleLoading(false);
            
            // Fermer toutes les modales
            document.querySelectorAll('.modal').forEach(modal => {
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) bsModal.hide();
            });
            
            // Recharger le tableau
            if (typeof dataTable !== 'undefined') {
                dataTable.row(`[data-client-id="${clientId}"]`).remove().draw();
            }
            
            showToast('Client supprimé avec succès', 'success');
        }, 1000);
    }
}

/**
 * Suspendre un client
 */
function suspendClient(clientId) {
    const reason = prompt('Raison de la suspension :');
    if (reason && reason.trim()) {
        toggleLoading(true);
        
        setTimeout(() => {
            toggleLoading(false);
            showToast('Client suspendu avec succès', 'success');
            
            // Fermer modale et recharger
            document.querySelectorAll('.modal').forEach(modal => {
                const bsModal = bootstrap.Modal.getInstance(modal);
                if (bsModal) bsModal.hide();
            });
            
            if (typeof dataTable !== 'undefined') {
                dataTable.ajax.reload(null, false);
            }
        }, 1000);
    }
}

/**
 * Activer un client
 */
function activateClient(clientId) {
    toggleLoading(true);
    
    setTimeout(() => {
        toggleLoading(false);
        showToast('Client activé avec succès', 'success');
        
        if (typeof dataTable !== 'undefined') {
            dataTable.ajax.reload(null, false);
        }
    }, 1000);
}

/**
 * Envoyer une notification
 */
function sendNotification(clientId) {
    console.log('Envoyer notification au client:', clientId);
    showToast('Notification envoyée avec succès', 'success');
}

/**
 * Voir l'historique
 */
function viewHistory(clientId) {
    console.log('Voir historique du client:', clientId);
    viewClient(clientId); // Ouvre la modale avec l'onglet historique
}

/**
 * Gérer le wallet
 */
function manageWallet(clientId) {
    console.log('Gérer wallet du client:', clientId);
    showToast('Fonctionnalité de gestion wallet en cours de développement', 'info');
}

// ============================================
// BULK ACTIONS
// ============================================

function bulkAction(action) {
    if (selectedClients.length === 0) {
        showToast('Aucun client sélectionné', 'warning');
        return;
    }
    
    const confirmMessage = {
        activate: `Activer ${selectedClients.length} client(s) ?`,
        deactivate: `Désactiver ${selectedClients.length} client(s) ?`,
        suspend: `Suspendre ${selectedClients.length} client(s) ?`,
        export: null
    };
    
    if (confirmMessage[action] && !confirm(confirmMessage[action])) {
        return;
    }
    
    toggleLoading(true);
    
    setTimeout(() => {
        toggleLoading(false);
        
        switch(action) {
            case 'export':
                exportSelectedClients();
                break;
            default:
                showToast(`Action "${action}" effectuée sur ${selectedClients.length} client(s)`, 'success');
                
                // Décocher tous les checkboxes
                $('.row-checkbox').prop('checked', false);
                $('#selectAll').prop('checked', false);
                selectedClients = [];
                $('#bulkActionsBar').slideUp(300);
                
                // Recharger le tableau
                if (typeof dataTable !== 'undefined') {
                    dataTable.ajax.reload(null, false);
                }
        }
    }, 1500);
}

function exportSelectedClients() {
    console.log('Export des clients sélectionnés:', selectedClients);
    showToast(`Export de ${selectedClients.length} client(s) en cours...`, 'info');
    
    // Ici vous pouvez implémenter l'export réel
    // Par exemple, générer un CSV des clients sélectionnés
}

// ============================================
// EXPORT DATA
// ============================================

function exportData() {
    const buttons = document.querySelector('.dt-buttons');
    if (buttons) {
        // Afficher temporairement les boutons
        buttons.style.display = 'flex';
        buttons.style.gap = '0.5rem';
        buttons.style.marginBottom = '1rem';
        
        // Les cacher après 10 secondes
        setTimeout(() => {
            buttons.style.display = 'none';
        }, 10000);
    }
}

// ============================================
// UTILITIES
// ============================================

function timeAgo(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const seconds = Math.floor((now - date) / 1000);
    
    const intervals = {
        année: 31536000,
        mois: 2592000,
        jour: 86400,
        heure: 3600,
        minute: 60
    };
    
    for (const [name, secondsInInterval] of Object.entries(intervals)) {
        const interval = Math.floor(seconds / secondsInInterval);
        if (interval >= 1) {
            return `Il y a ${interval} ${name}${interval > 1 && name !== 'mois' ? 's' : ''}`;
        }
    }
    
    return 'À l\'instant';
}

// Add CSS for rotation animation
const style = document.createElement('style');
style.textContent = `
    .rotate-animation {
        animation: rotate 1s linear infinite;
    }
    
    @keyframes rotate {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }
`;
document.head.appendChild(style);