<?php
$pageTitle = 'Aperçu Financier';
$breadcrumbs = [
    ['label' => 'Dashboard', 'url' => '../index.php'],
    ['label' => 'Finances', 'url' => '#'],
    ['label' => 'Aperçu', 'url' => '#']
];

require_once __DIR__ . '/../../config/constants.php';
require_once __DIR__ . '/../../config/functions.php';

// Charger les données mock
$interventionsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/interventions.json'), true);
$mechanicsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/mechanics.json'), true);
$statsData = json_decode(file_get_contents(__DIR__ . '/../../mockdata/stats.json'), true);

$interventions = $interventionsData['interventions'];
$mechanics = $mechanicsData['mechanics'];
$overview = $statsData['overview'];

// Calculs financiers
$todayRevenue = array_reduce($interventions, function($sum, $int) {
    return $sum + ($int['status'] === 'completed' && date('Y-m-d', strtotime($int['completed_at'])) === date('Y-m-d') ? $int['amount'] : 0);
}, 0);

$thisWeekRevenue = array_reduce($interventions, function($sum, $int) {
    return $sum + ($int['status'] === 'completed' && date('W', strtotime($int['completed_at'])) === date('W') ? $int['amount'] : 0);
}, 0);

$thisMonthRevenue = array_reduce($interventions, function($sum, $int) {
    return $sum + ($int['status'] === 'completed' && date('m', strtotime($int['completed_at'])) === date('m') ? $int['amount'] : 0);
}, 0);

$thisYearRevenue = array_reduce($interventions, function($sum, $int) {
    return $sum + ($int['status'] === 'completed' && date('Y', strtotime($int['completed_at'])) === date('Y') ? $int['amount'] : 0);
}, 0);

// Données pour les graphiques
$monthlyRevenue = [
    'labels' => ['Jan', 'Fév', 'Mar', 'Avr', 'Mai', 'Jun', 'Jul', 'Aoû', 'Sep', 'Oct', 'Nov', 'Déc'],
    'data' => [8500000, 9200000, 8800000, 9500000, 10200000, 9800000, 10500000, 11200000, 10800000, 11500000, 11800000, 12200000]
];

$paymentMethods = [
    'labels' => ['Portefeuille MECASOS', 'Carte bancaire', 'Espèces'],
    'data' => [65, 25, 10]
];

// Top earners (mécaniciens)
$topEarners = array_map(function($mechanic) {
    return [
        'name' => $mechanic['name'],
        'earnings' => $mechanic['total_earned'],
        'interventions' => $mechanic['total_interventions'],
        'rating' => $mechanic['rating']
    ];
}, array_slice($mechanics, 0, 5));

// Pending payments (simulé)
$pendingPayments = array_filter($mechanics, function($mechanic) {
    return $mechanic['pending_commission'] > 0;
});
?>

<!-- Page Header -->
<div class="page-header">
    <div class="page-header-content">
        <h1 class="page-title">
            <i class="bi bi-graph-up"></i>
            Aperçu Financier
        </h1>
        <p class="page-subtitle">Analyse des revenus et commissions MECASOS</p>
    </div>
    <div class="page-header-actions">
        <div class="date-selector">
            <select class="form-select" id="periodSelector">
                <option value="today">Aujourd'hui</option>
                <option value="week" selected>Cette semaine</option>
                <option value="month">Ce mois</option>
                <option value="year">Cette année</option>
            </select>
        </div>
        <button class="btn btn-outline-primary btn-icon-text" onclick="exportReport()">
            <i class="bi bi-download"></i>
            <span>Exporter</span>
        </button>
        <button class="btn btn-primary btn-icon-text" onclick="generateReport()">
            <i class="bi bi-file-earmark-text"></i>
            <span>Rapport</span>
        </button>
    </div>
</div>

<!-- Revenue Cards -->
<div class="row g-4 mb-4">
    <div class="col-xl-3 col-lg-6">
        <div class="finance-card revenue-card" data-aos="fade-up" data-aos-delay="0">
            <div class="card-body">
                <div class="card-icon bg-success">
                    <i class="bi bi-calendar-day"></i>
                </div>
                <div class="card-content">
                    <span class="card-label">Aujourd'hui</span>
                    <h3 class="card-value" data-target="<?= $todayRevenue ?>">
                        <span class="counter">0</span>
                        <small>FCFA</small>
                    </h3>
                    <div class="card-trend trend-up">
                        <i class="bi bi-arrow-up"></i>
                        <span>+12.5%</span>
                        <small>vs hier</small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-lg-6">
        <div class="finance-card revenue-card" data-aos="fade-up" data-aos-delay="100">
            <div class="card-body">
                <div class="card-icon bg-info">
                    <i class="bi bi-calendar-week"></i>
                </div>
                <div class="card-content">
                    <span class="card-label">Cette semaine</span>
                    <h3 class="card-value" data-target="<?= $thisWeekRevenue ?>">
                        <span class="counter">0</span>
                        <small>FCFA</small>
                    </h3>
                    <div class="card-trend trend-up">
                        <i class="bi bi-arrow-up"></i>
                        <span>+8.3%</span>
                        <small>vs sem. dernière</small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-lg-6">
        <div class="finance-card revenue-card" data-aos="fade-up" data-aos-delay="200">
            <div class="card-body">
                <div class="card-icon bg-warning">
                    <i class="bi bi-calendar-month"></i>
                </div>
                <div class="card-content">
                    <span class="card-label">Ce mois</span>
                    <h3 class="card-value" data-target="<?= $thisMonthRevenue ?>">
                        <span class="counter">0</span>
                        <small>FCFA</small>
                    </h3>
                    <div class="card-trend trend-up">
                        <i class="bi bi-arrow-up"></i>
                        <span>+15.7%</span>
                        <small>vs mois dernier</small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-lg-6">
        <div class="finance-card revenue-card" data-aos="fade-up" data-aos-delay="300">
            <div class="card-body">
                <div class="card-icon bg-gold">
                    <i class="bi bi-calendar-year"></i>
                </div>
                <div class="card-content">
                    <span class="card-label">Cette année</span>
                    <h3 class="card-value" data-target="<?= $thisYearRevenue ?>">
                        <span class="counter">0</span>
                        <small>FCFA</small>
                    </h3>
                    <div class="card-trend trend-up">
                        <i class="bi bi-arrow-up"></i>
                        <span>+22.4%</span>
                        <small>vs année dernière</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Charts Row -->
<div class="row g-4 mb-4">
    <!-- Revenue Chart -->
    <div class="col-xl-8">
        <div class="dashboard-card">
            <div class="card-header">
                <div class="card-header-content">
                    <h5 class="card-title">
                        <i class="bi bi-bar-chart-line"></i>
                        Évolution des Revenus
                    </h5>
                    <p class="card-subtitle">Revenus mensuels sur les 12 derniers mois</p>
                </div>
                <div class="card-header-actions">
                    <div class="btn-group" role="group">
                        <button type="button" class="btn btn-sm btn-outline-secondary active">12 Mois</button>
                        <button type="button" class="btn btn-sm btn-outline-secondary">6 Mois</button>
                        <button type="button" class="btn btn-sm btn-outline-secondary">3 Mois</button>
                    </div>
                </div>
            </div>
            <div class="card-body">
                <canvas id="revenueChart" height="80"></canvas>
            </div>
        </div>
    </div>

    <!-- Payment Methods Chart -->
    <div class="col-xl-4">
        <div class="dashboard-card">
            <div class="card-header">
                <div class="card-header-content">
                    <h5 class="card-title">
                        <i class="bi bi-pie-chart"></i>
                        Méthodes de Paiement
                    </h5>
                    <p class="card-subtitle">Répartition par méthode</p>
                </div>
            </div>
            <div class="card-body">
                <canvas id="paymentMethodsChart" height="200"></canvas>
            </div>
            <div class="card-footer">
                <div class="chart-legend">
                    <div class="legend-item">
                        <span class="legend-color" style="background: #E1A004;"></span>
                        <span class="legend-label">Portefeuille MECASOS</span>
                        <span class="legend-value">65%</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-color" style="background: #10B981;"></span>
                        <span class="legend-label">Carte bancaire</span>
                        <span class="legend-value">25%</span>
                    </div>
                    <div class="legend-item">
                        <span class="legend-color" style="background: #3B82F6;"></span>
                        <span class="legend-label">Espèces</span>
                        <span class="legend-value">10%</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Bottom Row -->
<div class="row g-4">
    <!-- Top Earners -->
    <div class="col-xl-6">
        <div class="dashboard-card">
            <div class="card-header">
                <div class="card-header-content">
                    <h5 class="card-title">
                        <i class="bi bi-trophy"></i>
                        Top Rémunérateurs
                    </h5>
                    <p class="card-subtitle">Mécaniciens les mieux rémunérés</p>
                </div>
                <div class="card-header-actions">
                    <a href="../users/mechanics.php" class="btn btn-sm btn-outline-primary">
                        Voir tous
                    </a>
                </div>
            </div>
            <div class="card-body">
                <div class="top-earners-list">
                    <?php foreach($topEarners as $index => $earner): ?>
                    <div class="earner-item">
                        <div class="earner-rank rank-<?= $index + 1 ?>">
                            <?= $index + 1 ?>
                        </div>
                        <div class="earner-info">
                            <h6 class="earner-name"><?= e($earner['name']) ?></h6>
                            <div class="earner-stats">
                                <span class="stat-item">
                                    <i class="bi bi-tools"></i>
                                    <?= $earner['interventions'] ?> interventions
                                </span>
                                <span class="stat-item">
                                    <i class="bi bi-star-fill text-warning"></i>
                                    <?= $earner['rating'] ?>
                                </span>
                            </div>
                        </div>
                        <div class="earner-amount">
                            <span class="amount-value"><?= formatCurrency($earner['earnings']) ?></span>
                            <small class="text-muted">total</small>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Pending Payments -->
    <div class="col-xl-6">
        <div class="dashboard-card">
            <div class="card-header">
                <div class="card-header-content">
                    <h5 class="card-title">
                        <i class="bi bi-clock-history"></i>
                        Paiements en Attente
                    </h5>
                    <p class="card-subtitle">Commissions à verser aux mécaniciens</p>
                </div>
                <div class="card-header-actions">
                    <button class="btn btn-sm btn-primary" onclick="processAllPayments()">
                        Traiter tout
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="pending-payments-list">
                    <?php foreach(array_slice($pendingPayments, 0, 5) as $mechanic): ?>
                    <div class="payment-item">
                        <div class="payment-info">
                            <img src="<?= getDefaultAvatar($mechanic['name']) ?>" alt="" class="mechanic-avatar-sm">
                            <div class="payment-details">
                                <span class="mechanic-name"><?= e($mechanic['name']) ?></span>
                                <small class="text-muted">
                                    <?= $mechanic['total_interventions'] ?> interventions •
                                    Commission: <?= formatCurrency($mechanic['pending_commission']) ?>
                                </small>
                            </div>
                        </div>
                        <div class="payment-actions">
                            <button class="btn btn-sm btn-outline-success" onclick="processPayment(<?= $mechanic['id'] ?>)">
                                <i class="bi bi-check-circle"></i>
                                Payer
                            </button>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <?php if(empty($pendingPayments)): ?>
                <div class="text-center py-4">
                    <i class="bi bi-check-circle text-success" style="font-size: 3rem;"></i>
                    <h6 class="mt-3">Tous les paiements sont à jour</h6>
                    <p class="text-muted">Aucune commission en attente de versement</p>
                </div>
                <?php endif; ?>
            </div>
            <div class="card-footer">
                <div class="payment-summary">
                    <div class="summary-item">
                        <span class="summary-label">Total en attente:</span>
                        <span class="summary-value text-warning">
                            <?= formatCurrency(array_sum(array_column($pendingPayments, 'pending_commission'))) ?>
                        </span>
                    </div>
                    <div class="summary-item">
                        <span class="summary-label">Prochaine échéance:</span>
                        <span class="summary-value">15 Décembre 2025</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// ========================================
// FINANCE OVERVIEW CHARTS
// ========================================
document.addEventListener('DOMContentLoaded', function() {
    // Counter animations
    const counters = document.querySelectorAll('.counter');
    counters.forEach(counter => {
        const target = parseInt(counter.getAttribute('data-target'));
        const duration = 2000;
        const increment = target / (duration / 16);
        let current = 0;

        const updateCounter = () => {
            current += increment;
            if (current < target) {
                counter.textContent = Math.floor(current).toLocaleString('fr-FR');
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = target.toLocaleString('fr-FR');
            }
        };

        setTimeout(updateCounter, 300);
    });

    // Revenue Chart
    const revenueCtx = document.getElementById('revenueChart');
    if (revenueCtx) {
        const revenueData = <?= json_encode($monthlyRevenue) ?>;

        new Chart(revenueCtx, {
            type: 'line',
            data: {
                labels: revenueData.labels,
                datasets: [{
                    label: 'Revenus (FCFA)',
                    data: revenueData.data,
                    borderColor: '#E1A004',
                    backgroundColor: 'rgba(225, 160, 4, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4,
                    pointBackgroundColor: '#E1A004',
                    pointBorderColor: '#1D1D1B',
                    pointBorderWidth: 2,
                    pointRadius: 6,
                    pointHoverRadius: 8,
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        backgroundColor: '#1A1A1A',
                        titleColor: '#E1A004',
                        bodyColor: '#FFFFFF',
                        borderColor: '#E1A004',
                        borderWidth: 1,
                        padding: 12,
                        displayColors: false,
                        callbacks: {
                            label: function(context) {
                                return context.parsed.y.toLocaleString('fr-FR') + ' FCFA';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            color: '#A0A0A0',
                            callback: function(value) {
                                return (value / 1000000).toFixed(1) + 'M';
                            }
                        },
                        grid: {
                            color: 'rgba(255, 255, 255, 0.05)',
                            drawBorder: false
                        }
                    },
                    x: {
                        ticks: {
                            color: '#A0A0A0'
                        },
                        grid: {
                            display: false
                        }
                    }
                },
                interaction: {
                    intersect: false,
                    mode: 'index'
                }
            }
        });
    }

    // Payment Methods Chart
    const paymentCtx = document.getElementById('paymentMethodsChart');
    if (paymentCtx) {
        const paymentData = <?= json_encode($paymentMethods) ?>;

        new Chart(paymentCtx, {
            type: 'doughnut',
            data: {
                labels: paymentData.labels,
                datasets: [{
                    data: paymentData.data,
                    backgroundColor: [
                        '#E1A004',
                        '#10B981',
                        '#3B82F6'
                    ],
                    borderWidth: 3,
                    borderColor: '#1A1A1A',
                    hoverBorderWidth: 4,
                    hoverBorderColor: '#E1A004'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                cutout: '70%',
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        backgroundColor: '#1A1A1A',
                        titleColor: '#E1A004',
                        bodyColor: '#FFFFFF',
                        borderColor: '#E1A004',
                        borderWidth: 1,
                        padding: 12,
                        callbacks: {
                            label: function(context) {
                                return context.label + ': ' + context.parsed + '%';
                            }
                        }
                    }
                }
            }
        });
    }
});

// Functions
function exportReport() {
    showNotification('Export du rapport en cours...', 'info');
    setTimeout(() => {
        showNotification('Rapport exporté avec succès!', 'success');
    }, 2000);
}

function generateReport() {
    showNotification('Génération du rapport détaillé...', 'info');
    setTimeout(() => {
        showNotification('Rapport généré!', 'success');
    }, 3000);
}

function processPayment(mechanicId) {
    showNotification('Traitement du paiement...', 'info');
    setTimeout(() => {
        showNotification('Paiement effectué avec succès!', 'success');
    }, 1500);
}

function processAllPayments() {
    showNotification('Traitement de tous les paiements...', 'info');
    setTimeout(() => {
        showNotification('Tous les paiements ont été traités!', 'success');
    }, 3000);
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
        if(notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}
</script>

<style>
/* Finance Cards */
.finance-card {
    background: var(--card-background);
    border: 1px solid var(--border-color);
    border-radius: 12px;
    overflow: hidden;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}

.finance-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
}

.finance-card .card-body {
    padding: 24px;
    display: flex;
    align-items: center;
    gap: 16px;
}

.card-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 24px;
    flex-shrink: 0;
}

.card-content {
    flex: 1;
}

.card-label {
    color: var(--text-secondary);
    font-size: 14px;
    margin-bottom: 4px;
}

.card-value {
    color: var(--text-primary);
    font-size: 24px;
    font-weight: 700;
    margin: 0;
    display: flex;
    align-items: baseline;
    gap: 8px;
}

.card-value small {
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 400;
}

.card-trend {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 12px;
    margin-top: 8px;
}

.trend-up {
    color: #10B981;
}

.trend-down {
    color: #EF4444;
}

.card-trend small {
    color: var(--text-secondary);
}

/* Top Earners */
.top-earners-list {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.earner-item {
    display: flex;
    align-items: center;
    gap: 16px;
    padding: 16px;
    background: rgba(255, 255, 255, 0.02);
    border-radius: 8px;
}

.earner-rank {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 18px;
    color: white;
}

.rank-1 { background: linear-gradient(135deg, #FFD700, #FFA500); }
.rank-2 { background: linear-gradient(135deg, #C0C0C0, #A8A8A8); }
.rank-3 { background: linear-gradient(135deg, #CD7F32, #A0522D); }
.rank-4 { background: var(--primary-color); }
.rank-5 { background: var(--border-color); }

.earner-info {
    flex: 1;
}

.earner-name {
    margin: 0 0 8px 0;
    color: var(--text-primary);
}

.earner-stats {
    display: flex;
    gap: 16px;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 4px;
    font-size: 12px;
    color: var(--text-secondary);
}

.earner-amount {
    text-align: right;
}

.amount-value {
    display: block;
    color: var(--primary-color);
    font-weight: 600;
    font-size: 16px;
}

/* Pending Payments */
.pending-payments-list {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.payment-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 12px 16px;
    background: rgba(255, 255, 255, 0.02);
    border-radius: 8px;
}

.payment-info {
    display: flex;
    align-items: center;
    gap: 12px;
}

.mechanic-avatar-sm {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    object-fit: cover;
}

.payment-details {
    display: flex;
    flex-direction: column;
}

.mechanic-name {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 2px;
}

.payment-actions {
    flex-shrink: 0;
}

/* Chart Legend */
.chart-legend {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 8px;
}

.legend-color {
    width: 12px;
    height: 12px;
    border-radius: 50%;
}

.legend-label {
    flex: 1;
    font-size: 14px;
    color: var(--text-secondary);
}

.legend-value {
    font-weight: 500;
    color: var(--text-primary);
}

/* Payment Summary */
.payment-summary {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.summary-item {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 4px;
}

.summary-label {
    font-size: 12px;
    color: var(--text-secondary);
}

.summary-value {
    font-weight: 600;
    color: var(--text-primary);
}

/* Date Selector */
.date-selector {
    margin-right: 16px;
}

.date-selector .form-select {
    background: var(--card-background);
    border: 1px solid var(--border-color);
    color: var(--text-primary);
    padding: 8px 16px;
    border-radius: 8px;
    min-width: 150px;
}

.date-selector .form-select:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 0.2rem rgba(225, 160, 4, 0.25);
}

/* Responsive */
@media (max-width: 768px) {
    .page-header-actions {
        flex-direction: column;
        gap: 12px;
    }

    .date-selector {
        margin-right: 0;
        margin-bottom: 8px;
    }

    .finance-card .card-body {
        padding: 16px;
        flex-direction: column;
        text-align: center;
        gap: 12px;
    }

    .card-icon {
        width: 50px;
        height: 50px;
        font-size: 20px;
    }

    .earner-item {
        flex-direction: column;
        text-align: center;
        gap: 12px;
    }

    .payment-item {
        flex-direction: column;
        gap: 12px;
        text-align: center;
    }

    .payment-summary {
        flex-direction: column;
        gap: 16px;
    }
}
</style>
