<?php
/**
 * Wallet API Endpoints
 */

require_once __DIR__ . '/../cors.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../utils.php';

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance()->getConnection();

// ============================================
// GET /api/wallet - Get user's wallet
// ============================================
if ($method === 'GET' && !strpos($_SERVER['REQUEST_URI'], '/transactions')) {
    $userData = requireAuth();
    
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ?");
    $stmt->execute([$userData['userId']]);
    $wallet = $stmt->fetch();
    
    if (!$wallet) {
        sendResponse(false, 'Wallet not found', null, 404);
    }
    
    sendResponse(true, 'Wallet retrieved', ['wallet' => $wallet]);
}

// ============================================
// POST /api/wallet/topup - Add funds to wallet
// ============================================
if ($method === 'POST' && strpos($_SERVER['REQUEST_URI'], '/topup') !== false) {
    $userData = requireAuth();
    $input = getJsonInput();
    
    // Validate required fields
    if (!isset($input['amount']) || empty($input['paymentMethod'])) {
        sendResponse(false, 'Amount and payment method are required', null, 400);
    }
    
    $amount = floatval($input['amount']);
    if ($amount <= 0) {
        sendResponse(false, 'Amount must be greater than 0', null, 400);
    }
    
    // Validate payment method
    $validMethods = ['mixxByYas', 'moovMoney', 'card'];
    if (!in_array($input['paymentMethod'], $validMethods)) {
        sendResponse(false, 'Invalid payment method', null, 400);
    }
    
    // Get wallet
    $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ?");
    $stmt->execute([$userData['userId']]);
    $wallet = $stmt->fetch();
    
    if (!$wallet) {
        sendResponse(false, 'Wallet not found', null, 404);
    }
    
    // Start transaction
    $db->beginTransaction();
    
    try {
        // Update wallet balance
        $newBalance = $wallet['balance'] + $amount;
        $stmt = $db->prepare("UPDATE wallets SET balance = ?, updated_at = NOW() WHERE id = ?");
        $stmt->execute([$newBalance, $wallet['id']]);
        
        // Create transaction record
        $transactionId = generateUUID();
        $reference = 'TOP-' . strtoupper(substr(md5(uniqid()), 0, 10));
        
        $stmt = $db->prepare("
            INSERT INTO wallet_transactions (id, user_id, amount, transaction_type, payment_method, reference, description, created_at)
            VALUES (?, ?, ?, 'topup', ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $transactionId,
            $userData['userId'],
            $amount,
            $input['paymentMethod'],
            $reference,
            'Wallet top-up via ' . $input['paymentMethod']
        ]);
        
        $db->commit();
        
        // Get updated wallet
        $stmt = $db->prepare("SELECT * FROM wallets WHERE user_id = ?");
        $stmt->execute([$userData['userId']]);
        $wallet = $stmt->fetch();
        
        sendResponse(true, 'Wallet topped up successfully', [
            'wallet' => $wallet,
            'transaction' => [
                'id' => $transactionId,
                'amount' => $amount,
                'reference' => $reference
            ]
        ]);
    } catch (Exception $e) {
        $db->rollBack();
        sendResponse(false, 'Transaction failed', null, 500);
    }
}

// ============================================
// GET /api/wallet/transactions - Get wallet transactions
// ============================================
if ($method === 'GET' && strpos($_SERVER['REQUEST_URI'], '/transactions') !== false) {
    $userData = requireAuth();
    
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
    $offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;
    
    $stmt = $db->prepare("
        SELECT * FROM wallet_transactions 
        WHERE user_id = ? 
        ORDER BY created_at DESC 
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$userData['userId'], $limit, $offset]);
    $transactions = $stmt->fetchAll();
    
    sendResponse(true, 'Transactions retrieved', ['transactions' => $transactions]);
}

// Default response
sendResponse(false, 'Endpoint not found', null, 404);
