<?php
/**
 * Vehicles API Endpoints
 */

require_once __DIR__ . '/../cors.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../utils.php';

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance()->getConnection();

// ============================================
// GET /api/vehicles - Get vehicles for current user
// ============================================
if ($method === 'GET' && !preg_match('/\/\w+$/', $_SERVER['REQUEST_URI'])) {
    $userData = requireAuth();
    
    $userId = $_GET['userId'] ?? $userData['userId'];
    
    $stmt = $db->prepare("SELECT * FROM vehicles WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$userId]);
    $vehicles = $stmt->fetchAll();
    
    sendResponse(true, 'Vehicles retrieved', ['vehicles' => $vehicles]);
}

// ============================================
// GET /api/vehicles/:id - Get vehicle by ID
// ============================================
if ($method === 'GET' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $vehicleId = $matches[1];
    
    $stmt = $db->prepare("SELECT * FROM vehicles WHERE id = ?");
    $stmt->execute([$vehicleId]);
    $vehicle = $stmt->fetch();
    
    if (!$vehicle) {
        sendResponse(false, 'Vehicle not found', null, 404);
    }
    
    // Check if user owns this vehicle
    if ($vehicle['user_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    sendResponse(true, 'Vehicle retrieved', ['vehicle' => $vehicle]);
}

// ============================================
// POST /api/vehicles - Create new vehicle
// ============================================
if ($method === 'POST') {
    $userData = requireAuth();
    $input = getJsonInput();
    
    // Validate required fields
    if (empty($input['brand']) || empty($input['model']) || empty($input['year'])) {
        sendResponse(false, 'Brand, model, and year are required', null, 400);
    }
    
    // Validate vehicle type
    $vehicleType = $input['vehicleType'] ?? 'car';
    if (!in_array($vehicleType, ['car', 'motorcycle'])) {
        sendResponse(false, 'Invalid vehicle type', null, 400);
    }
    
    $vehicleId = generateUUID();
    
    $stmt = $db->prepare("
        INSERT INTO vehicles (id, user_id, brand, model, year, plate_number, vehicle_type, created_at, updated_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
    ");
    
    $stmt->execute([
        $vehicleId,
        $userData['userId'],
        sanitizeInput($input['brand']),
        sanitizeInput($input['model']),
        intval($input['year']),
        isset($input['plateNumber']) ? sanitizeInput($input['plateNumber']) : null,
        $vehicleType
    ]);
    
    // Get created vehicle
    $stmt = $db->prepare("SELECT * FROM vehicles WHERE id = ?");
    $stmt->execute([$vehicleId]);
    $vehicle = $stmt->fetch();
    
    sendResponse(true, 'Vehicle created', ['vehicle' => $vehicle], 201);
}

// ============================================
// PUT /api/vehicles/:id - Update vehicle
// ============================================
if ($method === 'PUT' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $vehicleId = $matches[1];
    
    // Check if vehicle exists and user owns it
    $stmt = $db->prepare("SELECT user_id FROM vehicles WHERE id = ?");
    $stmt->execute([$vehicleId]);
    $vehicle = $stmt->fetch();
    
    if (!$vehicle) {
        sendResponse(false, 'Vehicle not found', null, 404);
    }
    
    if ($vehicle['user_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    $input = getJsonInput();
    
    $updateFields = [];
    $params = [];
    
    if (isset($input['brand'])) {
        $updateFields[] = "brand = ?";
        $params[] = sanitizeInput($input['brand']);
    }
    
    if (isset($input['model'])) {
        $updateFields[] = "model = ?";
        $params[] = sanitizeInput($input['model']);
    }
    
    if (isset($input['year'])) {
        $updateFields[] = "year = ?";
        $params[] = intval($input['year']);
    }
    
    if (isset($input['plateNumber'])) {
        $updateFields[] = "plate_number = ?";
        $params[] = sanitizeInput($input['plateNumber']);
    }
    
    if (isset($input['vehicleType'])) {
        if (!in_array($input['vehicleType'], ['car', 'motorcycle'])) {
            sendResponse(false, 'Invalid vehicle type', null, 400);
        }
        $updateFields[] = "vehicle_type = ?";
        $params[] = $input['vehicleType'];
    }
    
    if (empty($updateFields)) {
        sendResponse(false, 'No fields to update', null, 400);
    }
    
    $updateFields[] = "updated_at = NOW()";
    $params[] = $vehicleId;
    
    $sql = "UPDATE vehicles SET " . implode(', ', $updateFields) . " WHERE id = ?";
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    
    // Get updated vehicle
    $stmt = $db->prepare("SELECT * FROM vehicles WHERE id = ?");
    $stmt->execute([$vehicleId]);
    $vehicle = $stmt->fetch();
    
    sendResponse(true, 'Vehicle updated', ['vehicle' => $vehicle]);
}

// ============================================
// DELETE /api/vehicles/:id - Delete vehicle
// ============================================
if ($method === 'DELETE' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $vehicleId = $matches[1];
    
    // Check if vehicle exists and user owns it
    $stmt = $db->prepare("SELECT user_id FROM vehicles WHERE id = ?");
    $stmt->execute([$vehicleId]);
    $vehicle = $stmt->fetch();
    
    if (!$vehicle) {
        sendResponse(false, 'Vehicle not found', null, 404);
    }
    
    if ($vehicle['user_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    $stmt = $db->prepare("DELETE FROM vehicles WHERE id = ?");
    $stmt->execute([$vehicleId]);
    
    sendResponse(true, 'Vehicle deleted');
}

// Default response
sendResponse(false, 'Endpoint not found', null, 404);
