<?php
/**
 * Breakdown Requests API Endpoints
 */

require_once __DIR__ . '/../cors.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../utils.php';

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance()->getConnection();

// ============================================
// GET /api/requests - Get breakdown requests
// ============================================
if ($method === 'GET' && !preg_match('/\/\w+$/', $_SERVER['REQUEST_URI'])) {
    $userData = requireAuth();
    
    $status = $_GET['status'] ?? null;
    $userType = $userData['userType'];
    
    $sql = "SELECT br.*, 
            u.name as client_name, u.phone as client_phone,
            v.brand, v.model, v.year, v.plate_number,
            m.name as mechanic_name, m.phone as mechanic_phone
            FROM breakdown_requests br
            LEFT JOIN users u ON br.client_id = u.id
            LEFT JOIN vehicles v ON br.vehicle_id = v.id
            LEFT JOIN users m ON br.mechanic_id = m.id
            WHERE 1=1";
    
    $params = [];
    
    // Filter based on user type
    if ($userType === 'client') {
        $sql .= " AND br.client_id = ?";
        $params[] = $userData['userId'];
    } else if ($userType === 'mechanic') {
        // Mechanics see their requests or pending requests
        $sql .= " AND (br.mechanic_id = ? OR br.status = 'pending')";
        $params[] = $userData['userId'];
    }
    
    if ($status) {
        $sql .= " AND br.status = ?";
        $params[] = $status;
    }
    
    $sql .= " ORDER BY br.created_at DESC";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $requests = $stmt->fetchAll();
    
    sendResponse(true, 'Requests retrieved', ['requests' => $requests]);
}

// ============================================
// GET /api/requests/:id - Get request by ID
// ============================================
if ($method === 'GET' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $requestId = $matches[1];
    
    $stmt = $db->prepare("
        SELECT br.*, 
            u.name as client_name, u.phone as client_phone, u.email as client_email,
            v.brand, v.model, v.year, v.plate_number, v.vehicle_type,
            m.name as mechanic_name, m.phone as mechanic_phone
        FROM breakdown_requests br
        LEFT JOIN users u ON br.client_id = u.id
        LEFT JOIN vehicles v ON br.vehicle_id = v.id
        LEFT JOIN users m ON br.mechanic_id = m.id
        WHERE br.id = ?
    ");
    $stmt->execute([$requestId]);
    $request = $stmt->fetch();
    
    if (!$request) {
        sendResponse(false, 'Request not found', null, 404);
    }
    
    sendResponse(true, 'Request retrieved', ['request' => $request]);
}

// ============================================
// POST /api/requests - Create new breakdown request
// ============================================
if ($method === 'POST' && !preg_match('/\/\w+\/\w+$/', $_SERVER['REQUEST_URI'])) {
    $userData = requireAuth();
    
    // Only clients can create requests
    if ($userData['userType'] !== 'client') {
        sendResponse(false, 'Only clients can create requests', null, 403);
    }
    
    $input = getJsonInput();
    
    // Validate required fields
    if (empty($input['vehicleId']) || empty($input['breakdownType']) || 
        empty($input['description']) || empty($input['latitude']) || 
        empty($input['longitude']) || empty($input['address'])) {
        sendResponse(false, 'All fields are required', null, 400);
    }
    
    // Validate breakdown type
    $validTypes = ['battery', 'tire', 'fuelEmpty', 'towing', 'keyLost', 'engineFailure', 
                   'brakeFailure', 'electrical', 'overheating', 'starter', 'transmission', 
                   'suspension', 'lights', 'other'];
    
    if (!in_array($input['breakdownType'], $validTypes)) {
        sendResponse(false, 'Invalid breakdown type', null, 400);
    }
    
    // Verify vehicle belongs to user
    $stmt = $db->prepare("SELECT id FROM vehicles WHERE id = ? AND user_id = ?");
    $stmt->execute([$input['vehicleId'], $userData['userId']]);
    if (!$stmt->fetch()) {
        sendResponse(false, 'Vehicle not found or unauthorized', null, 404);
    }
    
    $requestId = generateUUID();
    
    $stmt = $db->prepare("
        INSERT INTO breakdown_requests (
            id, client_id, vehicle_id, breakdown_type, description, 
            latitude, longitude, address, status, created_at, updated_at
        )
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW(), NOW())
    ");
    
    $stmt->execute([
        $requestId,
        $userData['userId'],
        $input['vehicleId'],
        $input['breakdownType'],
        sanitizeInput($input['description']),
        floatval($input['latitude']),
        floatval($input['longitude']),
        sanitizeInput($input['address'])
    ]);
    
    // Get created request
    $stmt = $db->prepare("
        SELECT br.*, 
            v.brand, v.model, v.year, v.plate_number
        FROM breakdown_requests br
        LEFT JOIN vehicles v ON br.vehicle_id = v.id
        WHERE br.id = ?
    ");
    $stmt->execute([$requestId]);
    $request = $stmt->fetch();
    
    sendResponse(true, 'Request created', ['request' => $request], 201);
}

// ============================================
// PUT /api/requests/:id/accept - Mechanic accepts request
// ============================================
if ($method === 'PUT' && preg_match('/\/([a-zA-Z0-9-]+)\/accept$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    
    // Only mechanics can accept requests
    if ($userData['userType'] !== 'mechanic') {
        sendResponse(false, 'Only mechanics can accept requests', null, 403);
    }
    
    $requestId = $matches[1];
    
    // Check if request exists and is pending
    $stmt = $db->prepare("SELECT * FROM breakdown_requests WHERE id = ? AND status = 'pending'");
    $stmt->execute([$requestId]);
    $request = $stmt->fetch();
    
    if (!$request) {
        sendResponse(false, 'Request not found or already accepted', null, 404);
    }
    
    // Update request
    $stmt = $db->prepare("UPDATE breakdown_requests SET mechanic_id = ?, status = 'accepted', updated_at = NOW() WHERE id = ?");
    $stmt->execute([$userData['userId'], $requestId]);
    
    // Get updated request
    $stmt = $db->prepare("
        SELECT br.*, 
            u.name as client_name, u.phone as client_phone,
            v.brand, v.model, v.year
        FROM breakdown_requests br
        LEFT JOIN users u ON br.client_id = u.id
        LEFT JOIN vehicles v ON br.vehicle_id = v.id
        WHERE br.id = ?
    ");
    $stmt->execute([$requestId]);
    $request = $stmt->fetch();
    
    sendResponse(true, 'Request accepted', ['request' => $request]);
}

// ============================================
// PUT /api/requests/:id/status - Update request status
// ============================================
if ($method === 'PUT' && preg_match('/\/([a-zA-Z0-9-]+)\/status$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $requestId = $matches[1];
    $input = getJsonInput();
    
    if (empty($input['status'])) {
        sendResponse(false, 'Status is required', null, 400);
    }
    
    $validStatuses = ['pending', 'accepted', 'inProgress', 'completed', 'cancelled'];
    if (!in_array($input['status'], $validStatuses)) {
        sendResponse(false, 'Invalid status', null, 400);
    }
    
    // Get request
    $stmt = $db->prepare("SELECT * FROM breakdown_requests WHERE id = ?");
    $stmt->execute([$requestId]);
    $request = $stmt->fetch();
    
    if (!$request) {
        sendResponse(false, 'Request not found', null, 404);
    }
    
    // Authorization check
    if ($userData['userType'] === 'client' && $request['client_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    if ($userData['userType'] === 'mechanic' && $request['mechanic_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    // Update status
    $stmt = $db->prepare("UPDATE breakdown_requests SET status = ?, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$input['status'], $requestId]);
    
    // If status is completed and price is provided, update price
    if ($input['status'] === 'completed' && isset($input['price'])) {
        $stmt = $db->prepare("UPDATE breakdown_requests SET price = ? WHERE id = ?");
        $stmt->execute([floatval($input['price']), $requestId]);
    }
    
    // Get updated request
    $stmt = $db->prepare("SELECT * FROM breakdown_requests WHERE id = ?");
    $stmt->execute([$requestId]);
    $request = $stmt->fetch();
    
    sendResponse(true, 'Request status updated', ['request' => $request]);
}

// ============================================
// DELETE /api/requests/:id - Cancel/Delete request
// ============================================
if ($method === 'DELETE' && preg_match('/\/([a-zA-Z0-9-]+)$/', $_SERVER['REQUEST_URI'], $matches)) {
    $userData = requireAuth();
    $requestId = $matches[1];
    
    // Get request
    $stmt = $db->prepare("SELECT * FROM breakdown_requests WHERE id = ?");
    $stmt->execute([$requestId]);
    $request = $stmt->fetch();
    
    if (!$request) {
        sendResponse(false, 'Request not found', null, 404);
    }
    
    // Only client who created the request can delete it
    if ($request['client_id'] !== $userData['userId']) {
        sendResponse(false, 'Unauthorized', null, 403);
    }
    
    // Can only cancel pending or accepted requests
    if (!in_array($request['status'], ['pending', 'accepted'])) {
        sendResponse(false, 'Cannot cancel request in current status', null, 400);
    }
    
    // Update status to cancelled instead of deleting
    $stmt = $db->prepare("UPDATE breakdown_requests SET status = 'cancelled', updated_at = NOW() WHERE id = ?");
    $stmt->execute([$requestId]);
    
    sendResponse(true, 'Request cancelled');
}

// Default response
sendResponse(false, 'Endpoint not found', null, 404);
