<?php
/**
 * Authentication API Endpoints
 */

require_once __DIR__ . '/../cors.php';
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../utils.php';
require_once __DIR__ . '/../email_service.php';

$method = $_SERVER['REQUEST_METHOD'];
$db = Database::getInstance()->getConnection();

// ============================================
// POST /api/auth/register - Register new user
// ============================================
if ($method === 'POST' && strpos($_SERVER['REQUEST_URI'], '/register') !== false) {
    $input = getJsonInput();
    
    // Validate required fields
    if (empty($input['name']) || empty($input['email']) || empty($input['password']) || empty($input['phone']) || empty($input['userType'])) {
        sendResponse(false, 'All fields are required', null, 400);
    }
    
    // Validate email
    if (!isValidEmail($input['email'])) {
        sendResponse(false, 'Invalid email address', null, 400);
    }
    
    // Validate user type
    if (!in_array($input['userType'], ['client', 'mechanic'])) {
        sendResponse(false, 'Invalid user type', null, 400);
    }
    
    // Check if email already exists
    $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$input['email']]);
    if ($stmt->fetch()) {
        sendResponse(false, 'Email already registered', null, 409);
    }
    
    // Create user
    $userId = generateUUID();
    $passwordHash = hashPassword($input['password']);
    
    $stmt = $db->prepare("
        INSERT INTO users (id, name, email, phone, password_hash, user_type, is_verified, created_at, updated_at)
        VALUES (?, ?, ?, ?, ?, ?, 1, NOW(), NOW())
    ");
    
    $stmt->execute([
        $userId,
        sanitizeInput($input['name']),
        sanitizeInput($input['email']),
        sanitizeInput($input['phone']),
        $passwordHash,
        $input['userType']
    ]);
    
    // Create wallet for user
    $walletId = generateUUID();
    $stmt = $db->prepare("INSERT INTO wallets (id, user_id, balance, created_at, updated_at) VALUES (?, ?, 0.00, NOW(), NOW())");
    $stmt->execute([$walletId, $userId]);
    
    // Generate token
    $token = generateJWT($userId, $input['userType']);
    
    // Send welcome email
    $emailService = new EmailService();
    $emailService->sendWelcomeEmail($input['email'], $input['name'], $input['userType']);
    
    // Get user data
    $stmt = $db->prepare("SELECT id, name, email, phone, user_type, is_online, created_at, updated_at FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    sendResponse(true, 'Registration successful', [
        'user' => $user,
        'token' => $token
    ], 201);
}

// ============================================
// POST /api/auth/login - Login user
// ============================================
if ($method === 'POST' && strpos($_SERVER['REQUEST_URI'], '/login') !== false) {
    $input = getJsonInput();
    
    // Validate required fields
    if (empty($input['email']) || empty($input['password'])) {
        sendResponse(false, 'Email and password are required', null, 400);
    }
    
    // Get user
    $stmt = $db->prepare("SELECT * FROM users WHERE email = ?");
    $stmt->execute([sanitizeInput($input['email'])]);
    $user = $stmt->fetch();
    
    if (!$user || !verifyPassword($input['password'], $user['password_hash'])) {
        sendResponse(false, 'Invalid credentials', null, 401);
    }
    
    // Generate token
    $token = generateJWT($user['id'], $user['user_type']);
    
    // Update online status
    $stmt = $db->prepare("UPDATE users SET is_online = 1, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$user['id']]);
    
    // Remove password hash from response
    unset($user['password_hash']);
    $user['is_online'] = 1;
    
    sendResponse(true, 'Login successful', [
        'user' => $user,
        'token' => $token
    ]);
}

// ============================================
// POST /api/auth/logout - Logout user
// ============================================
if ($method === 'POST' && strpos($_SERVER['REQUEST_URI'], '/logout') !== false) {
    $userData = requireAuth();
    
    // Update online status
    $stmt = $db->prepare("UPDATE users SET is_online = 0, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$userData['userId']]);
    
    sendResponse(true, 'Logout successful');
}

// ============================================
// POST /api/auth/forgot-password - Request password reset
// ============================================
if ($method === 'POST' && strpos($_SERVER['REQUEST_URI'], '/forgot-password') !== false) {
    $input = getJsonInput();
    
    if (empty($input['email'])) {
        sendResponse(false, 'Email is required', null, 400);
    }
    
    // Check if user exists
    $stmt = $db->prepare("SELECT id, name FROM users WHERE email = ?");
    $stmt->execute([sanitizeInput($input['email'])]);
    $user = $stmt->fetch();
    
    if (!$user) {
        // Don't reveal if email exists or not
        sendResponse(true, 'If the email exists, a reset link will be sent');
    }
    
    // Generate reset token
    $resetToken = bin2hex(random_bytes(32));
    $expiresAt = date('Y-m-d H:i:s', strtotime('+1 hour'));
    
    // Store reset token
    $stmt = $db->prepare("INSERT INTO password_resets (email, token, expires_at) VALUES (?, ?, ?)");
    $stmt->execute([$input['email'], $resetToken, $expiresAt]);
    
    // Send reset email
    $emailService = new EmailService();
    $emailService->sendPasswordResetEmail($input['email'], $resetToken);
    
    sendResponse(true, 'If the email exists, a reset link will be sent');
}

// ============================================
// GET /api/auth/me - Get current user
// ============================================
if ($method === 'GET' && strpos($_SERVER['REQUEST_URI'], '/me') !== false) {
    $userData = requireAuth();
    
    $stmt = $db->prepare("SELECT id, name, email, phone, user_type, is_online, profile_picture, address, latitude, longitude, created_at, updated_at FROM users WHERE id = ?");
    $stmt->execute([$userData['userId']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        sendResponse(false, 'User not found', null, 404);
    }
    
    sendResponse(true, 'User retrieved', ['user' => $user]);
}

// Default response
sendResponse(false, 'Endpoint not found', null, 404);
