# MECASOS Backend - Server Configuration

## Apache .htaccess Configuration

If using Apache, create this `.htaccess` file in the `/api` directory.

```apache
# Enable rewrite engine
RewriteEngine On

# Force HTTPS (recommended for production)
# RewriteCond %{HTTPS} off
# RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]

# Handle CORS preflight requests
RewriteCond %{REQUEST_METHOD} OPTIONS
RewriteRule ^(.*)$ $1 [R=200,L]

# Route all requests to index.php
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php [QSA,L]

# Prevent direct access to PHP files (except index.php)
RewriteCond %{REQUEST_FILENAME} -f
RewriteCond %{REQUEST_URI} !index\.php$
RewriteCond %{REQUEST_URI} \.php$
RewriteRule . - [F,L]
```

## Nginx Configuration

If using Nginx, add this to your server block:

```nginx
server {
    listen 80;
    server_name mecasos.numerexdigital.com;
    root /path/to/mecasos/backend;
    index index.php;

    # Force HTTPS (recommended for production)
    # return 301 https://$server_name$request_uri;

    location /api {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }

    # Prevent access to hidden files
    location ~ /\. {
        deny all;
    }

    # Handle CORS
    add_header 'Access-Control-Allow-Origin' '*' always;
    add_header 'Access-Control-Allow-Methods' 'GET, POST, PUT, DELETE, OPTIONS' always;
    add_header 'Access-Control-Allow-Headers' 'Content-Type, Authorization' always;
}
```

## PHP Requirements

Minimum PHP version: 7.4

Required extensions:
- PDO
- pdo_mysql
- json
- mbstring
- openssl

## Installation Steps

### 1. Upload Files
Upload all files from the `backend` folder to:
```
mecasos.numerexdigital.com/api/
```

Your directory structure should be:
```
/api/
├── index.php
├── config.php
├── database.php
├── utils.php
├── email_service.php
├── cors.php
├── api/
│   ├── auth.php
│   ├── users.php
│   ├── vehicles.php
│   ├── requests.php
│   ├── reviews.php
│   └── wallet.php
└── .htaccess (for Apache)
```

### 2. Install Dependencies

Install PHPMailer using Composer:
```bash
cd /path/to/api
composer require phpmailer/phpmailer
```

If Composer is not available, you can manually download PHPMailer and include it.

### 3. Database Setup

Import the SQL schema from `database_schema.md`:
```bash
mysql -u numerexd_db -p numerexd_db < database_schema.sql
```

Or use phpMyAdmin to import the SQL.

### 4. Configure PHP

Update `config.php`:
- Change `JWT_SECRET` to a secure random string
- Verify database credentials
- Verify SMTP credentials
- Set appropriate error reporting for production

### 5. Set Permissions

```bash
# For directories
chmod 755 /path/to/api
chmod 755 /path/to/api/api

# For PHP files
chmod 644 /path/to/api/*.php
chmod 644 /path/to/api/api/*.php

# Create uploads directory
mkdir -p /path/to/api/uploads
chmod 755 /path/to/api/uploads
```

### 6. Test the API

Visit: `https://mecasos.numerexdigital.com/api`

You should see a JSON response with available endpoints.

## Security Checklist

- [ ] Change JWT_SECRET to a random secure string
- [ ] Enable HTTPS
- [ ] Update CORS allowed origins in production
- [ ] Set display_errors to 0 in production
- [ ] Implement rate limiting
- [ ] Set up proper file permissions
- [ ] Use environment variables for sensitive data
- [ ] Enable PHP security extensions (e.g., disable_functions)
- [ ] Set up database backups
- [ ] Configure firewall rules
- [ ] Monitor error logs regularly

## Troubleshooting

### "Database connection failed"
- Verify database credentials in `config.php`
- Check if MySQL is running
- Ensure database exists and user has proper permissions

### "Email sending failed"
- Verify SMTP credentials
- Check if port 465 is open
- Test SMTP connection separately

### 404 errors on API endpoints
- Check if mod_rewrite is enabled (Apache)
- Verify .htaccess is uploaded
- Check Nginx configuration

### CORS errors
- Verify CORS headers in `cors.php`
- Check browser console for specific errors
- Ensure preflight requests are handled

## Monitoring

Monitor these log files:
- PHP error log: `/var/log/php/error.log`
- Apache/Nginx access log
- Apache/Nginx error log
- Application-specific logs (if implemented)

## Backup

Regular backups should include:
- Database (daily)
- Uploaded files
- Configuration files

```bash
# Database backup
mysqldump -u numerexd_db -p numerexd_db > backup_$(date +%Y%m%d).sql

# Files backup
tar -czf api_backup_$(date +%Y%m%d).tar.gz /path/to/api
```
